/* Name: main.c
 * Project: hid-data, example how to use HID for data transfer
 * Author: Christian Starkjohann
 * Creation Date: 2008-04-11
 * Tabsize: 4
 * Copyright: (c) 2008 by OBJECTIVE DEVELOPMENT Software GmbH
 * License: GNU GPL v2 (see License.txt) or proprietary (CommercialLicense.txt)
 * This Revision: $Id: main.c 592 2008-05-04 20:07:30Z cs $
 */

/*
This example should run on most AVRs with only little changes. No special
hardware resources except INT0 are used. You may have to change usbconfig.h for
different I/O pins for USB. Please note that USB D+ must be the INT0 pin, or
at least be connected to INT0 as well.
*/

#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>  /* for sei() */
#include <util/delay.h>     /* for _delay_ms() */
#include <avr/eeprom.h>

#include <avr/pgmspace.h>   /* required by usbdrv.h */
#include "usbdrv.h"
#include "oddebug.h"        /* This is also an example for using debug macros */


/* ------------------------------------------------------------------------- */
/* ----------------------------- USB interface ----------------------------- */
/* ------------------------------------------------------------------------- */
#define RP_Count 6

PROGMEM char usbHidReportDescriptor[22] = {    /* USB report descriptor */
    0x06, 0x00, 0xff,              // USAGE_PAGE (Vendor Defined Page  1)
    0x09, 0x01,                    // USAGE (Vendor Usage 1)
    0xa1, 0x01,                    // COLLECTION (Application)
    0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
    0x26, 0xff, 0x00,              //   LOGICAL_MAXIMUM (255)
    0x75, 0x08,                    //   REPORT_SIZE (8)
    0x95, RP_Count,                //   REPORT_COUNT (16) // (128) 0x80
    0x09, 0x00,                    //   USAGE (Undefined)
    0xb2, 0x02, 0x01,              //   FEATURE (Data,Var,Abs,Buf)
    0xc0                           // END_COLLECTION
};
/* Since we define only one feature report, we don't use report-IDs (which
 * would be the first byte of the report). The entire report consists of 128
 * opaque data bytes.
 */

/* The following variables store the status of the current data transfer */
static uchar    currentAddress;
static uchar    bytesRemaining;

		 
#define PIN_RC5 PINB0



/* usbFunctionWrite() is called when the host sends a chunk of data to the
 * device. For more information see the documentation in usbdrv/usbdrv.h.
 */
uchar  usbFunctionWrite(uchar *data, uchar len)
{
	
	if(bytesRemaining == 0)
        return 1;               /* end of transfer */
		
    if(len > bytesRemaining)
        len = bytesRemaining;
		
    /* Begin RC5 transmitt */
	if(len == 6)
	{
		uchar Bit_Pos   = 0x80;
		uchar Byte_Pos  = 2;
		uchar Bit_Count = data[0];
		uchar Byte_Send = data[1];
		uchar n;
	
		TCNT0  = 0;
		TCCR0A = (1<<WGM01);
		TCCR0B = (1<<CS01);
		do
		{
			
			if ( Byte_Send & Bit_Pos)
			{
				
				OCR0A = 54;
				for(n=0; n<32; n++)
				{
					TIFR  = (1<<OCF0A);
					while( !(TIFR & (1<<OCF0A)) );
				}
				
				for(n=0; n<32; n++)
				{
					OCR0A  = 27; // 27
					PORTB |= (1 << PIN_RC5);
					TIFR   = (1<<OCF0A);
					
					while(!(TIFR & (1<<OCF0A)));
					// ca. 13s
					
					OCR0A  = 28; // 28
					PORTB &= ~(1 << PIN_RC5);
					TIFR   = (1<<OCF0A);
					
					while( !(TIFR & (1<<OCF0A)) );
					// ca. 13s
				}
			}
			else
			{
				for(n=0; n<32; n++)
				{
					
					OCR0A = 27; // 27
					PORTB |=  (1 << PIN_RC5);
					TIFR  = (1<<OCF0A);				
					
					while( !(TIFR & (1<<OCF0A)) );
					// ca. 13s
					
					OCR0A = 28; // 28
					PORTB &= ~(1 << PIN_RC5);
					TIFR  = (1<<OCF0A);				
					
					while( !(TIFR & (1<<OCF0A)) );
					// ca. 13s
				}
				
				OCR0A = 54;
				for(n=0; n<32; n++)
				{
					TIFR  = (1<<OCF0A);
					while( !(TIFR & (1<<OCF0A)) );
				}
			
			}
			Bit_Pos >>= 1;
			
			if(Bit_Pos == 0)
			{
				Byte_Send = data[Byte_Pos++];
				Bit_Pos   = 0x80;
			}
		}
		while(--Bit_Count);
		
		TCCR0B = 0;
	
	}
	/* End RC5 transmitt */
	
	bytesRemaining = 0;
	

    return bytesRemaining == 0; /* return 1 if this was the last chunk */
}

/* ------------------------------------------------------------------------- */

usbMsgLen_t usbFunctionSetup(uchar data[8])
{
	usbRequest_t *rq = (void *)data;

    if((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS)    /* HID class request */
	{    
		if(rq->bRequest == USBRQ_HID_GET_REPORT)  /* wValue: ReportType (highbyte), ReportID (lowbyte) */
		{
			/* since we have only one report type, we can ignore the report-ID */
            bytesRemaining = RP_Count;
            currentAddress = 0;
            return USB_NO_MSG;  /* use usbFunctionRead() to obtain data */
        }
		else if(rq->bRequest == USBRQ_HID_SET_REPORT)
		{
            /* since we have only one report type, we can ignore the report-ID */
            bytesRemaining = RP_Count;
            currentAddress = 0;
            return USB_NO_MSG;  /* use usbFunctionWrite() to receive data from host */
        }
    }
	else
	{
        /* ignore vendor type requests, we don't use any */
    }
    return 0;
}


/* ------------------------------------------------------------------------- */
/* ------------------------ Oscillator Calibration ------------------------- */
/* ------------------------------------------------------------------------- */

/* Calibrate the RC oscillator to 8.25 MHz. The core clock of 16.5 MHz is
 * derived from the 66 MHz peripheral clock by dividing. Our timing reference
 * is the Start Of Frame signal (a single SE0 bit) available immediately after
 * a USB RESET. We first do a binary search for the OSCCAL value and then
 * optimize this value with a neighboorhod search.
 * This algorithm may also be used to calibrate the RC oscillator directly to
 * 12 MHz (no PLL involved, can therefore be used on almost ALL AVRs), but this
 * is wide outside the spec for the OSCCAL value and the required precision for
 * the 12 MHz clock! Use the RC oscillator calibrated to 12 MHz for
 * experimental purposes only!
 */
static void calibrateOscillator(void)
{
	uchar       step = 128;
	uchar       trialValue = 0, optimumValue;
	int         x, optimumDev, targetValue = (unsigned)(1499 * (double)F_CPU / 10.5e6 + 0.5);

    /* do a binary search: */
    do
	{
        OSCCAL = trialValue + step;
        x = usbMeasureFrameLength();    /* proportional to current real frequency */
        if(x < targetValue)             /* frequency still too low */
            trialValue += step;
        step >>= 1;
    }
	while(step > 0);

    /* We have a precision of +/- 1 for optimum OSCCAL here */
    /* now do a neighborhood search for optimum value */
    optimumValue = trialValue;
    optimumDev = x; /* this is certainly far away from optimum */
    for(OSCCAL = trialValue - 1; OSCCAL <= trialValue + 1; OSCCAL++)
	{
        x = usbMeasureFrameLength() - targetValue;
        if(x < 0)
            x = -x;
        if(x < optimumDev)
		{
            optimumDev = x;
            optimumValue = OSCCAL;
        }
    }
    OSCCAL = optimumValue;
}
/*
Note: This calibration algorithm may try OSCCAL values of up to 192 even if
the optimum value is far below 192. It may therefore exceed the allowed clock
frequency of the CPU in low voltage designs!
You may replace this search algorithm with any other algorithm you like if
you have additional constraints such as a maximum CPU clock.
For version 5.x RC oscillators (those with a split range of 2x128 steps, e.g.
ATTiny25, ATTiny45, ATTiny85), it may be useful to search for the optimum in
both regions.
*/

/* ------------------------------------------------------------------------- */

void usbEventResetReady(void)
{
    calibrateOscillator();
    //eeprom_write_byte((uchar *)0x7F, OSCCAL);   /* store the calibrated value in EEPROM */
}

/* ------------------------------------------------------------------------- */

int main(void)
{
	uchar   i;

    wdt_enable(WDTO_2S);
    /* Even if you don't use the watchdog, turn it off here. On newer devices,
     * the status of the watchdog (on/off, period) is PRESERVED OVER RESET!
     */
 
    /* RESET status: all port bits are inputs without pull-up.
     * That's the way we need D+ and D-. Therefore we don't need any
     * additional hardware initialization.
     */
	 
	/* Init RC5 Pin */
	PORTB &= ~(1<<PIN_RC5);
	DDRB  |=  (1<<PIN_RC5);


    usbInit();
    usbDeviceDisconnect();  /* enforce re-enumeration, do this while interrupts are disabled! */

    i = 0;
    while(--i)             /* fake USB disconnect for > 250 ms */
	{
		wdt_reset();
        _delay_ms(1);
    }
    usbDeviceConnect();
    
	sei();

    for(;;){                /* main event loop */
        wdt_reset();
        usbPoll();
    }
    return 0;
}

/* ------------------------------------------------------------------------- */

